<?php
// --- INICIO DE LÓGICA MULTI-USUARIO SEGURA ---
session_start();

$configFile = 'prompts_config.json';

// Define los prompts y claves por defecto (plantilla)
function getDefaultPrompts() {
    return [
        'titulo' => "Eres un experto en crear títulos optimizados para videos de YouTube. Tu tarea es ayudar al usuario a generar títulos claros, relevantes y optimizados para SEO, que despierten curiosidad sin ser engañosos. Analiza la información proporcionada por el usuario y genera un solo título por turno que refleje fielmente el contenido del video, sea sumamente claro, atractivo, contenga la palabra clave de forma natural (preferiblemente al inicio), tenga entre 60 y 70 caracteres, use capitalización legible, y, si añade valor, incluya un emoji sutil. No uses clickbait. No incluyas introducciones ni explicaciones: responde únicamente con el título. Mantén un tono conciso, profesional y preciso en la aplicación de estas reglas. Tema: \"\$tema\"",
        'descripcion' => "Eres un asistente experto en redacción de descripciones de videos de YouTube. Tu objetivo es ayudar a los usuarios a crear textos concisos, atractivos y optimizados para SEO, incorporando palabras clave relevantes de forma natural. Analiza la información proporcionada por el usuario y genera una descripción que comience con un resumen llamativo, detalle los puntos clave del video, sus beneficios y contexto adicional, todo en párrafos cortos y fáciles de leer. Si es relevante, sugiere timestamps. Incluye llamadas a la acción claras y finaliza con 3–5 hashtags específicos. Asegúrate de que la descripción sea honesta, útil y que optimice el video para las recomendaciones de YouTube, manteniendo un tono profesional, directo y enfocado en la funcionalidad. Queda prohibido agregar marcas de tiempo. Escribe en texto plano, está prohibido utilizar etiquetas Markdown de formato. Limítate a escribir la almohadilla de los hashtags sin una diagonal invertida antes, es decir escribe #, en vez de \\#. Tema: \"\$tema\"",
        'miniatura' => "Eres un generador de solicitudes para la creación de miniaturas de YouTube diseñadas en formato 16:9 y alta resolución (idealmente 1280x720 píxeles). Indica la resolución de la imagen y formato 16:9. Debes generar una solicitud concisa para un artista visual o un modelo de generación de imágenes. La solicitud debe comenzar con la frase exacta: \"Crea una imagen\". La imagen debe ser visualmente impactante, con colores brillantes y alto contraste, muy clara y enfocada, que incluya solo 2 o 3 elementos clave. En el lado derecho debe aparecer un elemento principal dominante: un objeto icónico y relevante para el tema del video o una cara muy expresiva. En el lado izquierdo, superpuesto sobre la imagen, debe haber una frase con verbos en presente simple y segunda persona, muy legible, conciso y con alto contraste, que resuma el qué se conseguirá con el video. El estilo general debe ser limpio, moderno y profesional, optimizado para generar clics de forma honesta y efectiva. Agrega algún ícono relevante al tema. La solicitud debe ser lo suficientemente descriptiva para que el artista o modelo comprenda completamente la visión deseada. Asegúrate de escribir en texto plano, sin ningún tipo de etiqueta de formato Markdown (está prohibido). Es obligatorio que todos los elementos de la plantilla sean muy de acuerdo al tema del video. Responde ahora. Tema: \"\$tema\"",
        'etiquetas' => "Eres un generador de etiquetas SEO para videos de YouTube. Al recibir el tema del video del usuario, identifica el tema principal como etiqueta base, crea sinónimos, frases relacionadas, etiquetas long-tail y algunas generales relevantes, asegurando que todas sean pertinentes al contenido. Une las etiquetas en una sola línea separadas por comas, sin exceder los 500 caracteres. Responde únicamente con esa línea, sin comentarios ni explicaciones. Sé directo, profesional y enfocado en la tarea. Asegúrate de no escribir punto final (está prohibido). Tema: \"\$tema\""
    ];
}

// Carga los prompts por defecto
$prompts = getDefaultPrompts();
// Si el usuario ya ha personalizado sus prompts, los usamos
if (isset($_SESSION['user_prompts'])) {
    $prompts = array_merge($prompts, $_SESSION['user_prompts']);
}

// Carga las claves de API desde la sesión del usuario
$api_keys = $_SESSION['api_keys'] ?? ['openai' => '', 'gemini' => ''];

// Lógica para guardar o restaurar ajustes
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    if ($_POST['action'] === 'save_settings') {
        // Guardar en la SESIÓN del usuario, no en el archivo JSON
        $_SESSION['api_keys']['openai'] = $_POST['openai_api_key'];
        $_SESSION['api_keys']['gemini'] = $_POST['gemini_api_key'];
        $_SESSION['user_prompts']['titulo'] = $_POST['prompt_titulo'];
        $_SESSION['user_prompts']['descripcion'] = $_POST['prompt_descripcion'];
        $_SESSION['user_prompts']['miniatura'] = $_POST['prompt_miniatura'];
        $_SESSION['user_prompts']['etiquetas'] = $_POST['prompt_etiquetas'];
    } elseif ($_POST['action'] === 'restore_defaults') {
        // Restaurar solo los prompts en la sesión del usuario
        $_SESSION['user_prompts'] = getDefaultPrompts();
    }
    header("Location: " . $_SERVER['PHP_SELF']);
    exit;
}

$OPENAI_API_KEY = $api_keys['openai'];
$GEMINI_API_KEY = $api_keys['gemini'];
$keys_are_set = !empty($OPENAI_API_KEY) && !empty($GEMINI_API_KEY);

// --- FIN DE LÓGICA MULTI-USUARIO SEGURA ---

function llamarOpenAI($prompt, $max_tokens = 100, $temperature = 0.7) {
    global $OPENAI_API_KEY;
    if (empty($OPENAI_API_KEY) || $OPENAI_API_KEY === 'sk-proj-xxxx') return "Error: La clave de API de OpenAI no está configurada.";

    $data = [
        "model" => "gpt-4o-mini",
        "messages" => [ ["role" => "user", "content" => $prompt] ],
        "max_tokens" => $max_tokens,
        "temperature" => $temperature
    ];
    $ch = curl_init("https://api.openai.com/v1/chat/completions");
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [ "Content-Type: application/json", "Authorization: Bearer $OPENAI_API_KEY" ]);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
    $response = curl_exec($ch);
    curl_close($ch);
    $result = json_decode($response, true);
    if (isset($result['choices'][0]['message']['content'])) {
        return trim($result['choices'][0]['message']['content']);
    }
    return "Error al generar contenido. Verifica tu clave de API de OpenAI en los Ajustes.";
}

function llamarGeminiImagen($prompt) {
    global $GEMINI_API_KEY;
    if (empty($GEMINI_API_KEY) || $GEMINI_API_KEY === 'AIzaSy-xxxx') return ['error' => 'Error: La clave de API de Gemini no está configurada.'];

    $GEMINI_ENDPOINT = 'https://generativelanguage.googleapis.com/v1beta/models/gemini-1.5-flash:generateContent?key=' . $GEMINI_API_KEY;
    
    $data = [ "contents" => [ [ "parts" => [ ["text" => $prompt] ] ] ] ];
    
    $ch = curl_init($GEMINI_ENDPOINT);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [ "Content-Type: application/json" ]);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
    $response = curl_exec($ch);
    curl_close($ch);
    $result = json_decode($response, true);

    if (isset($result['candidates'][0]['content']['parts'][0]['text'])) {
        $text_response = $result['candidates'][0]['content']['parts'][0]['text'];
        if(preg_match('/```json\s*(\{.*?\})\s*```/s', $text_response, $matches)) {
            $imageData = json_decode($matches[1], true);
             if (isset($imageData['imageB64'])) {
                return [ 'base64' => $imageData['imageB64'], 'mime' => $imageData['mimeType'] ?? 'image/png' ];
             }
        }
    }
    return [ 'error' => 'No se pudo generar la imagen. Respuesta de Gemini: ' . htmlspecialchars(substr($response, 0, 200)) ];
}
?>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <title>Generador de Contenidos para YouTube</title>
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <script src="https://cdn.tailwindcss.com"></script>
    <style>
        body { background: linear-gradient(135deg, #1e293b 0%, #0f172a 100%); min-height: 100vh; }
        .glass { background: rgba(255,255,255,0.08); backdrop-filter: blur(8px); border-radius: 1.5rem; box-shadow: 0 8px 32px 0 rgba(31, 38, 135, 0.37); }
        .tab-active { border-bottom: 2px solid #38bdf8; color: #38bdf8; }
        .modal-backdrop { background-color: rgba(0,0,0,0.6); backdrop-filter: blur(5px); }
        .disabled-overlay { position: absolute; inset: 0; background-color: rgba(15, 23, 42, 0.8); backdrop-filter: blur(2px); z-index: 10; border-radius: 1.5rem; display: flex; align-items: center; justify-content: center; text-align: center; padding: 1rem; }
    </style>
</head>
<body class="flex items-center justify-center min-h-screen text-white p-4">

    <div id="settings-modal" class="fixed inset-0 z-50 items-center justify-center hidden modal-backdrop">
        <div class="glass p-8 rounded-2xl w-full max-w-4xl max-h-[90vh] overflow-y-auto">
            <h2 class="text-2xl font-bold mb-6 text-sky-400">Ajustes</h2>
            <p class="text-slate-400 text-sm mb-4">Tus claves de API y prompts se guardarán de forma segura en tu sesión de navegador actual.</p>
            <form method="post">
                <input type="hidden" name="action" value="save_settings">
                
                <div class="grid grid-cols-1 md:grid-cols-2 gap-6 mb-6">
                    <div>
                        <label for="openai_api_key" class="block mb-2 text-sm font-medium text-slate-300">Clave API de OpenAI</label>
                        <input type="password" id="openai_api_key" name="openai_api_key" value="<?php echo htmlspecialchars($api_keys['openai']); ?>" class="w-full px-4 py-2 rounded-lg bg-slate-800 text-white focus:outline-none focus:ring-2 focus:ring-sky-400">
                    </div>
                    <div>
                        <label for="gemini_api_key" class="block mb-2 text-sm font-medium text-slate-300">Clave API de Gemini</label>
                        <input type="password" id="gemini_api_key" name="gemini_api_key" value="<?php echo htmlspecialchars($api_keys['gemini']); ?>" class="w-full px-4 py-2 rounded-lg bg-slate-800 text-white focus:outline-none focus:ring-2 focus:ring-sky-400">
                    </div>
                </div>

                <div class="space-y-4">
                    <div>
                        <label for="prompt_titulo" class="block mb-2 text-sm font-bold text-slate-300">Prompt para Títulos</label>
                        <textarea id="prompt_titulo" name="prompt_titulo" rows="4" class="w-full p-3 rounded-lg bg-slate-800 text-white text-sm focus:outline-none focus:ring-2 focus:ring-sky-400"><?php echo htmlspecialchars($prompts['titulo']); ?></textarea>
                    </div>
                    <div>
                        <label for="prompt_descripcion" class="block mb-2 text-sm font-bold text-slate-300">Prompt para Descripción</label>
                        <textarea id="prompt_descripcion" name="prompt_descripcion" rows="6" class="w-full p-3 rounded-lg bg-slate-800 text-white text-sm focus:outline-none focus:ring-2 focus:ring-sky-400"><?php echo htmlspecialchars($prompts['descripcion']); ?></textarea>
                    </div>
                    <div>
                        <label for="prompt_miniatura" class="block mb-2 text-sm font-bold text-slate-300">Prompt para Miniatura</label>
                        <textarea id="prompt_miniatura" name="prompt_miniatura" rows="6" class="w-full p-3 rounded-lg bg-slate-800 text-white text-sm focus:outline-none focus:ring-2 focus:ring-sky-400"><?php echo htmlspecialchars($prompts['miniatura']); ?></textarea>
                    </div>
                     <div>
                        <label for="prompt_etiquetas" class="block mb-2 text-sm font-bold text-slate-300">Prompt para Etiquetas</label>
                        <textarea id="prompt_etiquetas" name="prompt_etiquetas" rows="4" class="w-full p-3 rounded-lg bg-slate-800 text-white text-sm focus:outline-none focus:ring-2 focus:ring-sky-400"><?php echo htmlspecialchars($prompts['etiquetas']); ?></textarea>
                    </div>
                </div>

                <div class="flex items-center justify-between mt-8">
                    <button type="button" onclick="document.getElementById('restore-form').submit();" class="px-4 py-2 rounded-lg bg-amber-600 hover:bg-amber-700 font-semibold transition">Restaurar Prompts</button>
                    <div>
                        <button type="button" id="cancel-settings" class="px-6 py-2 rounded-lg bg-slate-600 hover:bg-slate-700 font-semibold transition mr-4">Cancelar</button>
                        <button type="submit" class="px-6 py-2 rounded-lg bg-sky-500 hover:bg-sky-600 font-semibold transition">Guardar Cambios</button>
                    </div>
                </div>
            </form>
            <form id="restore-form" method="post" class="hidden">
                 <input type="hidden" name="action" value="restore_defaults">
            </form>
        </div>
    </div>

    <div class="w-full max-w-3xl p-8 glass relative">
        <?php if (!$keys_are_set): ?>
            <div class="disabled-overlay">
                <div>
                    <p class="text-lg font-semibold text-white">Configuración Requerida</p>
                    <p class="text-slate-300 mb-4">Para usar la herramienta, por favor, introduce tus claves de API en los Ajustes.</p>
                    <button id="force-open-settings" class="px-6 py-2 rounded-lg bg-sky-500 hover:bg-sky-600 font-semibold transition">Ir a Ajustes</button>
                </div>
            </div>
        <?php endif; ?>
        
        <div class="flex justify-between items-center mb-4">
             <a href="index.php" class="inline-flex items-center text-sky-400 hover:text-sky-300 font-semibold transition">
                <span class="text-2xl mr-2">←</span> Volver al menú principal
            </a>
            <button id="open-settings" title="Ajustes">
                <svg xmlns="http://www.w3.org/2000/svg" class="h-6 w-6 text-slate-400 hover:text-sky-400 transition" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M10.325 4.317c.426-1.756 2.924-1.756 3.35 0a1.724 1.724 0 002.573 1.066c1.543-.94 3.31.826 2.37 2.37a1.724 1.724 0 001.065 2.572c1.756.426 1.756 2.924 0 3.35a1.724 1.724 0 00-1.066 2.573c.94 1.543-.826 3.31-2.37 2.37a1.724 1.724 0 00-2.572 1.065c-.426 1.756-2.924 1.756-3.35 0a1.724 1.724 0 00-2.573-1.066c-1.543.94-3.31-.826-2.37-2.37a1.724 1.724 0 00-1.065-2.572c-1.756-.426-1.756-2.924 0-3.35a1.724 1.724 0 001.066-2.573c-.94-1.543.826-3.31 2.37-2.37.996.608 2.296.07 2.572-1.065z" /><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 12a3 3 0 11-6 0 3 3 0 016 0z" /></svg>
            </button>
        </div>
        
        <h1 class="text-3xl font-bold mb-6 text-center">Generador de Contenidos para <span class="text-sky-400">YouTube</span></h1>
        
        <form method="post" action="<?php echo htmlspecialchars($_SERVER['PHP_SELF']); ?>" class="mb-8 flex flex-col md:flex-row gap-4 items-center justify-center">
            <input type="text" name="tema" placeholder="Introduce la temática del video..." required
                class="w-full md:w-2/3 px-4 py-2 rounded-lg bg-slate-800 text-white focus:outline-none focus:ring-2 focus:ring-sky-400 transition" 
                value="<?php echo isset($_POST['tema']) ? htmlspecialchars($_POST['tema']) : '' ?>">
            <button type="submit" class="px-6 py-2 rounded-lg bg-sky-500 hover:bg-sky-600 font-semibold transition">Generar</button>
        </form>

        <?php if (isset($_POST['tema']) && trim($_POST['tema']) !== ''): 
            $tema = trim($_POST['tema']);
        ?>
        <div>
            <div class="flex border-b border-slate-700 mb-6">
                <button class="tab-btn px-4 py-2 font-semibold focus:outline-none" data-tab="titulos">Títulos</button>
                <button class="tab-btn px-4 py-2 font-semibold focus:outline-none" data-tab="descripcion">Descripción</button>
                <button class="tab-btn px-4 py-2 font-semibold focus:outline-none" data-tab="miniatura">Miniatura</button>
                <button class="tab-btn px-4 py-2 font-semibold focus:outline-none" data-tab="etiquetas">Etiquetas</button>
            </div>
            
            <div id="tab-titulos" class="tab-content hidden">
                <h2 class="text-xl font-bold mb-2">Generador de Títulos</h2>
                <form method="post">
                    <input type="hidden" name="tema" value="<?php echo htmlspecialchars($tema); ?>">
                    <input type="hidden" name="generar" value="titulo">
                    <input type="hidden" name="tab_activa" value="titulos">
                    <button type="submit" class="px-4 py-2 bg-sky-500 rounded-lg hover:bg-sky-600 font-semibold">Generar Título</button>
                </form>
                <?php if (isset($_POST['generar']) && $_POST['generar'] === 'titulo'):
                    $promptTitulo = str_replace('$tema', $tema, $prompts['titulo']);
                    $titulo = llamarOpenAI($promptTitulo, 70, 0.7);
                    $titulo = trim($titulo, "\"'“”‘’");
                    echo '<div class="bg-slate-900 p-4 rounded-lg mt-2">'.htmlspecialchars($titulo).'</div>';
                endif; ?>
            </div>
            
            <div id="tab-descripcion" class="tab-content hidden">
                 <h2 class="text-xl font-bold mb-2">Generador de Descripción</h2>
                <form method="post">
                    <input type="hidden" name="tema" value="<?php echo htmlspecialchars($tema); ?>">
                    <input type="hidden" name="generar" value="descripcion">
                    <input type="hidden" name="tab_activa" value="descripcion">
                    <button type="submit" class="px-4 py-2 bg-sky-500 rounded-lg hover:bg-sky-600 font-semibold">Generar Descripción</button>
                </form>
                <?php if (isset($_POST['generar']) && $_POST['generar'] === 'descripcion'):
                    $promptDescripcion = str_replace('$tema', $tema, $prompts['descripcion']);
                    $descripcion = llamarOpenAI($promptDescripcion, 400, 0.7);
                    echo '<div class="bg-slate-900 p-4 rounded-lg mt-2 whitespace-pre-line">'.nl2br(htmlspecialchars($descripcion)).'</div>';
                endif; ?>
            </div>

            <div id="tab-miniatura" class="tab-content hidden">
                <h2 class="text-xl font-bold mb-2">Generador de Prompt para Miniatura</h2>
                <form method="post">
                    <input type="hidden" name="tema" value="<?php echo htmlspecialchars($tema); ?>">
                    <input type="hidden" name="generar" value="miniatura">
                    <input type="hidden" name="tab_activa" value="miniatura">
                    <button type="submit" class="px-4 py-2 bg-sky-500 rounded-lg hover:bg-sky-600 font-semibold">Generar Prompt</button>
                </form>
                <?php if (isset($_POST['generar']) && in_array($_POST['generar'], ['miniatura', 'gemini'])):
                    $promptGenerado = ($_POST['generar'] === 'miniatura') ? llamarOpenAI(str_replace('$tema', $tema, $prompts['miniatura']), 300, 0.7) : (isset($_POST['promptMiniatura']) ? $_POST['promptMiniatura'] : '');
                    
                    echo '<form method="post" class="flex flex-col gap-4 mt-4">';
                    echo '<label class="text-sm mb-1">Puedes editar el prompt antes de generar:</label>';
                    echo '<textarea name="promptMiniatura" rows="6" class="w-full p-2 rounded bg-slate-800 text-white mb-2">'.htmlspecialchars($promptGenerado).'</textarea>';
                    echo '<input type="hidden" name="tema" value="'.htmlspecialchars($tema).'">';
                    echo '<input type="hidden" name="tab_activa" value="miniatura">';
                    echo '<div class="flex gap-4">';
                    echo '<button type="submit" name="generar" value="miniatura" class="px-4 py-2 bg-sky-500 rounded-lg hover:bg-sky-600 font-semibold">Re-generar Prompt</button>';
                    echo '<button type="submit" name="generar" value="gemini" class="px-4 py-2 bg-green-500 rounded-lg hover:bg-green-600 font-semibold">Generar Imagen (Gemini)</button>';
                    echo '</div></form>';
                    
                    if ($_POST['generar'] === 'gemini' && !empty($promptGenerado)):
                        $imgResult = llamarGeminiImagen($promptGenerado);
                        if (isset($imgResult['base64'])) {
                            $src = "data:{$imgResult['mime']};base64,{$imgResult['base64']}";
                            echo '<div class="bg-slate-900 p-4 rounded-lg mt-2 flex flex-col items-center">';
                            echo '<img src="'.$src.'" alt="Miniatura generada" class="rounded-lg shadow-lg max-w-full h-auto mb-2" style="max-height:320px;">';
                            echo '<span class="text-xs text-slate-400">Imagen generada por Gemini</span>';
                            echo '</div>';
                        } else {
                            echo '<div class="bg-red-900 p-4 rounded-lg mt-2 text-red-200">'.($imgResult['error'] ?? 'Error desconocido').'</div>';
                        }
                    endif;
                endif; ?>
            </div>
            
            <div id="tab-etiquetas" class="tab-content hidden">
                <h2 class="text-xl font-bold mb-2">Generador de Etiquetas</h2>
                <form method="post">
                    <input type="hidden" name="tema" value="<?php echo htmlspecialchars($tema); ?>">
                    <input type="hidden" name="generar" value="etiquetas">
                    <input type="hidden" name="tab_activa" value="etiquetas">
                    <button type="submit" class="px-4 py-2 bg-sky-500 rounded-lg hover:bg-sky-600 font-semibold">Generar Etiquetas</button>
                </form>
                <?php if (isset($_POST['generar']) && $_POST['generar'] === 'etiquetas'):
                    $promptEtiquetas = str_replace('$tema', $tema, $prompts['etiquetas']);
                    $etiquetas = llamarOpenAI($promptEtiquetas, 200, 0.7);
                    echo '<div class="bg-slate-900 p-4 rounded-lg mt-2">'.htmlspecialchars($etiquetas).'</div>';
                endif; ?>
            </div>
        </div>
        <?php endif; ?>

        <script>
            function setupModal() {
                const modal = document.getElementById('settings-modal');
                const openBtn = document.getElementById('open-settings');
                const cancelBtn = document.getElementById('cancel-settings');
                const forceOpenBtn = document.getElementById('force-open-settings');

                const showModal = () => { modal.classList.remove('hidden'); modal.classList.add('flex'); };
                const hideModal = () => { modal.classList.add('hidden'); modal.classList.remove('flex'); };

                if (openBtn) openBtn.addEventListener('click', showModal);
                if (cancelBtn) cancelBtn.addEventListener('click', hideModal);
                if (forceOpenBtn) forceOpenBtn.addEventListener('click', showModal);
                if (modal) modal.addEventListener('click', (e) => { if (e.target === modal) hideModal(); });
                
                // Abrir modal automáticamente si las claves no están configuradas
                <?php if (!$keys_are_set): ?>
                showModal();
                <?php endif; ?>
            }
            setupModal();
            
            function setupTabs() {
                const tabBtns = document.querySelectorAll('.tab-btn');
                if (tabBtns.length === 0) return;

                const tabContents = document.querySelectorAll('.tab-content');
                
                function activateTab(tabId) {
                    tabBtns.forEach(b => b.classList.remove('tab-active'));
                    tabContents.forEach(tc => tc.classList.add('hidden'));
                    const btn = document.querySelector(`.tab-btn[data-tab="${tabId}"]`);
                    const content = document.getElementById(`tab-${tabId}`);
                    if(btn && content) {
                        btn.classList.add('tab-active');
                        content.classList.remove('hidden');
                    }
                    localStorage.setItem('tab_activa_yt_gen', tabId);
                }
                
                tabBtns.forEach(btn => {
                    btn.addEventListener('click', () => activateTab(btn.dataset.tab));
                });

                let activeTab = "<?php echo isset($_POST['tab_activa']) ? $_POST['tab_activa'] : ''; ?>";
                if (!activeTab) {
                    activeTab = localStorage.getItem('tab_activa_yt_gen') || 'titulos';
                }
                activateTab(activeTab);

                document.querySelectorAll('form').forEach(form => {
                    if(!form.id || !form.id.includes('settings-form')) {
                        const tabInput = form.querySelector('input[name="tab_activa"]');
                        if (tabInput) {
                            form.addEventListener('submit', function() {
                                tabInput.value = localStorage.getItem('tab_activa_yt_gen') || 'titulos';
                            });
                        }
                    }
                });
            }
            setupTabs();
        </script>
    </div>
</body>
</html>